package gov.va.genisis2.controller;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import gov.va.genisis2.bo.BusinessService;
import gov.va.genisis2.common.enums.WorkflowStatusEnum;
import gov.va.genisis2.dto.CommentHistoryDTO;
import gov.va.genisis2.dto.RequestDTO;
import gov.va.genisis2.dto.RequestHistoryDTO;
import gov.va.genisis2.dto.SourceDTO;
import gov.va.genisis2.dto.StudyApprovalDTO;
import gov.va.genisis2.dto.UserCountDTO;
import gov.va.genisis2.dto.UserDTO;
import gov.va.genisis2.dto.WorkflowStatusDTO;
import gov.va.genisis2.exceptions.ErrorEnum;
import gov.va.genisis2.exceptions.ErrorResponse;
import gov.va.genisis2.exceptions.GenisisException;
import gov.va.genisis2.model.Request;
import gov.va.genisis2.model.RequestType;
import gov.va.genisis2.util.rest.helper.ResponseWrapper;
import gov.va.genisis2.vo.CopyTableDomains;
import gov.va.genisis2.vo.CopyTableSource;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;

/**
 * The Class Genisis2Controller.
 */
@CrossOrigin(origins = "*")
@RestController
@RequestMapping("/services")
@Api("/services")
public class Genisis2Controller {
	
	/** The LOGGER. */
	private static final Logger LOGGER = LoggerFactory.getLogger(Genisis2Controller.class);
	
	/** The business service. */
	@Autowired
	private BusinessService businessService;

	/**
	 * 
	 * returns genisis2 API version
	 * 
	 * @return a json response
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/apiversion", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(hidden = true, value = "returns genisis2 API version")
	public ResponseEntity<ResponseWrapper> getGenisisApiVersion() throws GenisisException {
		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.getGenisisAPIDetails();
		} catch (Exception ex) {
			wrapper.setResponse(null);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_GENISISVERSION.getErrorMessage(), ex.getMessage(), ex);
		}
		return createReposneWrapper(wrapper);
	}
	
	/* T A B L E      C O P Y        T R A N S F E R      R E S T       C A L L S */
	/**
	 * 
	 * Performs the table copy script execution.
	 * 
	 * @param requestId
	 *            the request id of the data to be transferred
	 * @param copyTableSource
	 *            the source table name
	 * @return a json response
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/requests/{requestId}/copytable", method = RequestMethod.POST, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Performs the table copy script execution", notes = "This will copy table from VINCI using request id and copy table source", consumes = "application/json", produces = "application/json", response = String.class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = String.class) })
	public ResponseEntity<ResponseWrapper> performTableCopyTransfer(@ApiParam(required = true, name = "requestId", value = "Request Id") @PathVariable int requestId, @ApiParam(required = true, name = "copyTableSource", value = "Copy table source") @RequestBody CopyTableSource copyTableSource) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Table Copy Transfer.");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.performTableCopyTransfer(copyTableSource, requestId);
		} catch (Exception ex) {
			wrapper.setResponse(null);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_TABLECOPY.getErrorMessage(), ex.getMessage(), ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * @return a json response
	 * @throws GenisisException
	 *             throws GenisisException
	 */
	@RequestMapping(value = "/copytable/domains", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Copy table Domains", notes = "This call will get copy table domain from VINCI", consumes = "application/json", produces = "application/json", response = CopyTableDomains.class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = CopyTableDomains.class) })
	public ResponseEntity<ResponseWrapper> getCopyTableDomains() throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Copy Table Domains.");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.getCopyTableDomainDetails();
		} catch (Exception ex) {
			wrapper.setResponse(null);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_TABLECOPYDOMAINS.getErrorMessage(), ex.getMessage(), ex);
		}
		return createReposneWrapper(wrapper);
	}

	/* U S E R        M A N A G E ME N T       R E S T         C A L L S */
	/**
	 * Gets the user details by email.
	 * 
	 * @param email
	 *            the email
	 * @return a json response
	 * 
	 * @throws GenisisException   the Genisis exception
	 */
	@RequestMapping(value = "/users/{email}/email", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Gets the user details by email", notes = "Return information on a user with the email address", consumes = "application/json", produces = "application/json", response = UserDTO.class, tags = { "users" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = UserDTO.class) })
	public ResponseEntity<ResponseWrapper> getUserDetailsByEmail(@ApiParam(required = true, name = "email", value = "User's email address") @PathVariable String email) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info(" Get user by email.");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();
		
		try {
			wrapper = businessService.getUserDetailsByEmail(email);
		} catch (Exception ex) {
			wrapper.setResponse(email);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_USERBYEMAIL.getErrorMessage(), email, ex);
		}
		
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the user details by id.
	 *
	 * @param id
	 *            the id
	 * @return the user details by id
	 * 
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/users/{id:.+}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Gets the user details by ID", notes = "This call will retrieve information for the user with the id of researcher", consumes = "application/json", produces = "application/json", response = UserDTO.class, tags = { "users" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = UserDTO.class) })
	public ResponseEntity<ResponseWrapper> getUserDetailsById(@ApiParam(required = true, name = "id", value = "User's ID number") @PathVariable int id) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info(" Get user by Id.");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();
		
		try {
			wrapper = businessService.getUserDetailsById(id);
		} catch (Exception ex) {
			wrapper.setResponse(id);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_USERBYID.getErrorMessage(), id, ex);
		}
		
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the user role by uid.
	 *
	 * @param uid
	 *            the uid
	 * @return the user role by uid
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/users/role/{uid}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Gets the user role by uid", notes = "Return User Role information on a user ID", consumes = "application/json", produces = "application/json", response = String.class, tags = { "users" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = String.class) })
	public ResponseEntity<ResponseWrapper> getUserRoleByUid(@ApiParam(required = true, name = "userId", value = "User's ID number") @PathVariable int uid) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info(" Get user role by uid.");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();
		
		try {
			wrapper = businessService.getUserRole(uid);
		} catch (Exception ex) {
			wrapper.setResponse(uid);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_USERBYUID.getErrorMessage(), uid, ex);
		}
		
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the user details by VA username.
	 *
	 * @param username 
	 *            the VA User Name
	 * @return the user user by VA user name
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/users/username/{username}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Gets the user details by username", notes = "Return User information on a user name", consumes = "application/json", produces = "application/json", response = UserDTO.class, tags = { "users" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = UserDTO.class) })
	public ResponseEntity<ResponseWrapper> getUserByUsername(@ApiParam(required = true, name = "username", value = "VA User Name") @PathVariable String username) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info(" Get user details by username.");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();
		
		try {
			wrapper = businessService.getUserDetailsByUsername(username);
		} catch (Exception ex) {
			wrapper.setResponse(username);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_USERBYUSRNAME.getErrorMessage(), username, ex);
		}
		
		return createReposneWrapper(wrapper);
	}
	
	/**
	 * Gets the user counts by role and LDAP last refresh date/time
	 *
	 * @return the UserCountDTO user counts by role and LDAP last refresh date/time 
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/users/lastRefreshed", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Gets the user counts by role and LDAP last refresh date/time", notes = "Return the user counts by role and LDAP last refresh date/time", consumes = "application/json", produces = "application/json", response = UserCountDTO.class, tags = { "users" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = UserCountDTO.class) })
	public ResponseEntity<ResponseWrapper> getUserCountsAndLdapLastRefresh() throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info(" Get user counts by role and LDAP last refresh date/time");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();

		try {
			wrapper = businessService.getUserCountsAndLdapLastRefresh();
		} catch (Exception ex) {
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_LDAPLASTREFRESH.getErrorMessage(), ex);
		}
		
		return createReposneWrapper(wrapper);
	}
	
	/**
	 * refreshes Genisis2 user data
	 *
	 * @param username 
	 *            the VA User Name
	 * @return the UserCountDTO user counts by role and LDAP last refresh date/time
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/users/ldapRefresh/{username}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Refreshes Genisis2 user data", notes = "refreshes Genisis2 user data with VA LDAP user data", consumes = "application/json", produces = "application/json", response = UserCountDTO.class, tags = { "users" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = UserCountDTO.class) })
	public ResponseEntity<ResponseWrapper> refreshUserData(@ApiParam(required = true, name = "username", value = "VA User Name") @PathVariable String username) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("refresh Genisis2 user data");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();

		try {
			wrapper = businessService.refreshUserData(username);
		} catch (Exception ex) {
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.LDAP_USERREFRESH_FAILED.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_USERDATAREFRESH.getErrorMessage(), ex);
		}
		
		return createReposneWrapper(wrapper);
	}
	
	/* R E Q U E S T          R E S T           C A L L S */
	/**
	 * Creates the request.
	 *
	 * @param requestDto
	 *            the request
	 * @return the response entity
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/requests", method = RequestMethod.POST, headers = "Accept=*/*", consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Creates the request", notes = "This call will create a new request", consumes = "application/json", produces = "application/json", response = String.class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = String.class) })
	public ResponseEntity<ResponseWrapper> createRequest(@ApiParam(required = true, name = "request", value = "request details") @RequestBody RequestDTO requestDto) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Create Request ");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.createRequest(requestDto);
		} catch (Exception ex) {
			wrapper.setResponse(requestDto);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_CREATEREQUEST.getErrorMessage(), ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the all requests by study approval.
	 *
	 * @param studyApprovalId
	 *            the study approval id
	 * @return the all requests by study approval
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/requests/{studyApprovalId}/studyApproval", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Gets the all requests by study approval", notes = "Returns a list of requests that are linked to the study approval", consumes = "application/json", produces = "application/json", response = Request.class, tags = { "studyApprovals" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = RequestDTO.class) })
	public ResponseEntity<ResponseWrapper> getAllRequestsByStudyApproval(@ApiParam(required = true, name = "studyApprovalId", value = "Study Approval ID") @PathVariable int studyApprovalId) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getAllRequestsByStudyApproval ");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.getAllRequestsByStudyApproval(studyApprovalId);
		} catch (Exception ex) {
			wrapper.setResponse(null);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_GETREQUESTBYSA.getErrorMessage(), ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the requests by id.
	 *
	 * @param id
	 *            the id
	 * @return the requests by id
	 * 
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/requests/{id}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Gets the requests by id", notes = "Return a request given the request ID", consumes = "application/json", produces = "application/json", response = RequestDTO.class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = RequestDTO.class) })
	public ResponseEntity<ResponseWrapper> getRequestsById(@ApiParam(required = true, name = "id", value = "Request ID") @PathVariable int id) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Get Request by Id ");
		}

		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.getRequestByID(id);
		} catch (Exception ex) {
			wrapper.setResponse(null);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_GETREQUESTBYID.getErrorMessage(), id, ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the requests by user.
	 *
	 * @param uid
	 *            the uid
	 * @return the requests by user
	 * 
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/requests/{uid}/createdBy", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Gets the requests by user", notes = "Returns a list of all requests for a specific user", consumes = "application/json", produces = "application/json", response = RequestDTO.class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = RequestDTO.class) })
	public ResponseEntity<ResponseWrapper> getRequestsByUser(@ApiParam(required = true, name = "uid", value = "Request ID") @PathVariable String uid) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Get Requests by user ");
		}

		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.getRequestsByUID(uid);
		} catch (Exception ex) {
			wrapper.setResponse(null);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_GETREQUESTBYUID.getErrorMessage(), uid, ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the all requests.
	 *
	 * @return the all requests
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/requests", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Gets the all requests", notes = "Returns a list of all requests in the Genisis2", consumes = "application/json", produces = "application/json", response = Request.class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = Request.class) })
	public ResponseEntity<ResponseWrapper> getAllRequests() throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Get All Requests ");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.getAllRequests();
		} catch (Exception ex) {
			wrapper.setResponse(null);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_GETALLREQUESTS.getErrorMessage(), ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the request by data manager.
	 *
	 * @param id
	 *            the id
	 * @return the request by data manager
	 * 
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/requests/{id}/dataManager", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Gets the request by data manager", notes = "Returns a list of all requests created by or submitted to the data manager", consumes = "application/json", produces = "application/json", response = RequestDTO.class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = RequestDTO.class) })
	public ResponseEntity<ResponseWrapper> getRequestByDataManager(@ApiParam(required = true, name = "id", value = "User's ID") @PathVariable String id) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Request by DataManager  ");
		}

		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.getRequestsByDataManagers(id);
		} catch (Exception ex) {
			wrapper.setResponse(null);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_GETREQUESTSDDM.getErrorMessage(), id, ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the request by data source manager.
	 *
	 * @param id
	 *            the id
	 * @return the request by data manager
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/requests/{id}/dataSourceManager", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Gets the request by data source manager", notes = "Returns a list of all requests accepted or not-accepcted by all data source managers", consumes = "application/json", produces = "application/json", response = RequestDTO[].class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = RequestDTO[].class) })
	public ResponseEntity<ResponseWrapper> getRequestByDataSourceManager(@ApiParam(required = false, name = "id", value = "User's ID") @PathVariable String id) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Request by Data Source Manager  ");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.getRequestsDataSourceManagers(id);
		} catch (Exception ex) {
			wrapper.setResponse(null);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_GETREQUESTSDSM.getErrorMessage(), id, ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the request by status.
	 *
	 * @param status
	 *            the status
	 * @return the request by status
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/requests/{status}/status", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Gets the request by status", notes = "Returns a list of all requests with a specific status in the Genisis2", consumes = "application/json", produces = "application/json", response = RequestDTO[].class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = RequestDTO[].class) })
	public ResponseEntity<ResponseWrapper> getRequestByStatus(@ApiParam(required = false, name = "status", value = "The status of the request") @PathVariable String status) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Requests by Status /approved/status");
		}

		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.getAllRequestsByStaus(status);
		} catch (Exception ex) {
			wrapper.setResponse(null);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_GETREQUESTBYSTS.getErrorMessage(), status, ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Update request.
	 *
	 * @param id
	 *            the id
	 * @param requestDto
	 *            the request
	 * @return the response entity
	 * 
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/requests/{id}", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Update request details", notes = "Modify and save changes to a request details", consumes = "application/json", produces = "application/json", response = Integer.class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = Integer.class) })
	public ResponseEntity<ResponseWrapper> updateRequest(@ApiParam(required = true, name = "id", value = "RequestID") @PathVariable int id, @ApiParam(required = true, name = "request", value = "request details") @RequestBody RequestDTO requestDto) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Request Update");
		}

		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.persist(id, requestDto, WorkflowStatusEnum.UPDATE.getDesc());
		} catch (Exception ex) {
			wrapper.setResponse(requestDto);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_UPDATEREQUEST.getErrorMessage(), id, ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Submit request.
	 *
	 * @param id
	 *            the id
	 * @param requestDto
	 *            the request
	 * @return the response entity
	 * 
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/requests/{id}/submit", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Submit request", notes = "Modify and save changes to a request details", consumes = "application/json", produces = "application/json", response = Integer.class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = Integer.class) })
	public ResponseEntity<ResponseWrapper> submitRequest(@ApiParam(required = true, name = "id", value = "RequestID") @PathVariable int id, @ApiParam(required = true, name = "request", value = "request details") @RequestBody RequestDTO requestDto) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Request Submit ");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.persist(id, requestDto, WorkflowStatusEnum.SUBMITTED.getDesc());
		} catch (Exception ex) {
			wrapper.setResponse(requestDto);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_SUBMITREQUEST.getErrorMessage(), id, ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Sent request.
	 *
	 * @param id
	 *            the id
	 * @param requestDto
	 *            the request
	 * @return the response entity
	 * 
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/requests/{id}/sent", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Sent request", notes = "Modify and save changes to a request details", consumes = "application/json", produces = "application/json", response = Integer.class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = Integer.class) })
	public ResponseEntity<ResponseWrapper> sentRequest(@ApiParam(required = true, name = "id", value = "RequestID") @PathVariable int id, @ApiParam(required = true, name = "request", value = "request details") @RequestBody RequestDTO requestDto) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Request Sent/Approved ");
		}

		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.persist(id, requestDto, WorkflowStatusEnum.SENT.getDesc());
		} catch (Exception ex) {
			wrapper.setResponse(requestDto);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_SENTREQUEST.getErrorMessage(), id, ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Deny request.
	 *
	 * @param id
	 *            the id
	 * @param requestDto
	 *            the request
	 * @return the response entity
	 * 
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/requests/{id}/deny", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Deny request", notes = "Modify and save changes to a request details", consumes = "application/json", produces = "application/json", response = Integer.class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = Integer.class) })
	public ResponseEntity<ResponseWrapper> denyRequest(@ApiParam(required = true, name = "id", value = "RequestID") @PathVariable int id, @ApiParam(required = true, name = "request", value = "request details") @RequestBody RequestDTO requestDto) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Request Deny  ");
		}

		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.persist(id, requestDto, WorkflowStatusEnum.DENIED.getDesc());
		} catch (Exception ex) {
			wrapper.setResponse(requestDto);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_DENYREQUEST.getErrorMessage(), id, ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Return request.
	 *
	 * @param id
	 *            the id
	 * @param requestDto
	 *            the request
	 * @return the response entity
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/requests/{id}/return", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Return request", notes = "Modify and save changes to a request details", consumes = "application/json", produces = "application/json", response = Integer.class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = Integer.class) })
	public ResponseEntity<ResponseWrapper> returnRequest(@ApiParam(required = true, name = "id", value = "RequestID") @PathVariable int id, @ApiParam(required = true, name = "request", value = "request details") @RequestBody RequestDTO requestDto) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info(" Request Returned  ");
		}

		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.persist(id, requestDto, WorkflowStatusEnum.RETURNED.getDesc());
		} catch (Exception ex) {
			wrapper.setResponse(requestDto);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_RETURNREQUEST.getErrorMessage(), id, ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Fulfill request.
	 *
	 * @param id
	 *            the id
	 * @param requestDto
	 *            the request
	 * @return the response entity
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/requests/{id}/acceptdata", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Fulfill request", notes = "Modify and save changes to a request details", consumes = "application/json", produces = "application/json", response = Integer.class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = Integer.class) })
	public ResponseEntity<ResponseWrapper> fulfillRequest(@ApiParam(required = true, name = "id", value = "RequestID") @PathVariable int id, @ApiParam(required = true, name = "request", value = "request details") @RequestBody RequestDTO requestDto) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Accept Data");
		}

		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.persist(id, requestDto, WorkflowStatusEnum.REQUESTACCEPTED.getDesc());
		} catch (Exception ex) {
			wrapper.setResponse(requestDto);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_FULFILREQUEST.getErrorMessage(), id, ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Un fulfill request.
	 *
	 * @param id
	 *            the id
	 * @param requestDto
	 *            the request
	 * @return the response entity
	 * @throws GenisisException
	 *             the Genisis exception the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/denydata", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Un-fulfill request", notes = "Modify and save changes to a request details", consumes = "application/json", produces = "application/json", response = Integer.class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = Integer.class) })
	public ResponseEntity<ResponseWrapper> unFulfillRequest(@ApiParam(required = true, name = "id", value = "RequestID") @PathVariable int id, @ApiParam(required = true, name = "request", value = "request details") @RequestBody RequestDTO requestDto) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Deny Data");
		}

		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.persist(id, requestDto, WorkflowStatusEnum.REQUESTNOTACCEPTED.getDesc());
		} catch (Exception ex) {
			wrapper.setResponse(requestDto);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_UNFULFILREQUEST.getErrorMessage(), id, ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Reject data.
	 *
	 * @param id
	 *            the id
	 * @param requestDto
	 *            the request
	 * @return the response entity
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/requests/{id}/rejectdata", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Reject request", notes = "Modify and save changes to a request details", consumes = "application/json", produces = "application/json", response = Integer.class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = Integer.class) })
	public ResponseEntity<ResponseWrapper> rejectData(@ApiParam(required = true, name = "id", value = "RequestID") @PathVariable int id, @ApiParam(required = true, name = "request", value = "request details") @RequestBody RequestDTO requestDto) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Reject Results");
		}

		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.persist(id, requestDto, WorkflowStatusEnum.RESULTSNOTACCEPTED.getDesc());
		} catch (Exception ex) {
			wrapper.setResponse(requestDto);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_REJECTREQUEST.getErrorMessage(), id, ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Accept data.
	 *
	 * @param id
	 *            the id
	 * @param requestDto
	 *            the request
	 * @return the response entity
	 * 
	 * @throws GenisisException
	 *             the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/confirmdata", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Aceept request", notes = "Modify and save changes to a request details", consumes = "application/json", produces = "application/json", response = Integer.class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = Integer.class) })
	public ResponseEntity<ResponseWrapper> acceptData(@ApiParam(required = true, name = "id", value = "RequestID") @PathVariable int id, @ApiParam(required = true, name = "request", value = "request details") @RequestBody RequestDTO requestDto) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Accept Results");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.persist(id, requestDto, WorkflowStatusEnum.RESULTSACCEPTED.getDesc());
		} catch (Exception ex) {
			wrapper.setResponse(requestDto);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_ACCEPTDATA.getErrorMessage(), id, ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Request Deliver.
	 *
	 * @param id
	 *            the id
	 * @param requestDto
	 *            the request
	 * @return the response entity
	 * 
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/requests/{id}/resultsdelivered", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Request Deliver request", notes = "Modify and save changes to a request details", consumes = "application/json", produces = "application/json", response = Integer.class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = Integer.class) })
	public ResponseEntity<ResponseWrapper> requestDeliver(@ApiParam(required = true, name = "id", value = "RequestID") @PathVariable int id, @ApiParam(required = true, name = "request", value = "request details") @RequestBody RequestDTO requestDto) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Results Delivered");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.persist(id, requestDto, WorkflowStatusEnum.RESULTSDELIVERED.getDesc());
		} catch (Exception ex) {
			wrapper.setResponse(requestDto);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_DELIVERREQUEST.getErrorMessage(), id, ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * modifies request status
	 *
	 * @param id
	 *            the id
	 * @param operation
	 *            the operation
	 * @param requestDto
	 *            the request
	 * @return the response entity
	 * 
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/requests/{id}/{operation}", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Modify request status", notes = "Modifies request status", consumes = "application/json", produces = "application/json", response = Integer.class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = Integer.class) })
	public ResponseEntity<ResponseWrapper> updateRequestDetails(@ApiParam(required = true, name = "id", value = "RequestId") @PathVariable int id,
			@ApiParam(required = true, name = "operation", value = "action to be performed") @PathVariable String operation,
			@ApiParam(required = true, name = "request", value = "request details") @RequestBody RequestDTO requestDto) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Request {}", operation);
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.persist(id, requestDto, WorkflowStatusEnum.getStatus(operation));
		} catch (Exception ex) {
			wrapper.setResponse(requestDto);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_UPDATE_REQUEST.getErrorMessage(), operation, id, ex);
		}
		
		return createReposneWrapper(wrapper);
	}
	
	/**
	 * Gets the request status counts
	 *
	 * @return the WorkflowStatusDTO status counts
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/requests/status/count", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Gets the request status counts ", notes = "Return the Request Status counts", consumes = "application/json", produces = "application/json", response = WorkflowStatusDTO.class, tags = { "WorkflowStatus" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = WorkflowStatusDTO.class) })
	public ResponseEntity<ResponseWrapper> getRequestStatusCounts() throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info(" Get request status counts");
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.getRequestStatusCounts();
		} catch (Exception ex) {
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_REQUESTSTATUSCOUNTS.getErrorMessage(), ex);
		}
		return createReposneWrapper(wrapper);
	}
	
	/* S T U D Y        A P P R O V A L      R E S T       C A L L S */
	/**
	 * Creates the study approval.
	 *
	 * @param studyApprovalDto
	 *            the study approval
	 * @return the response entity
	 * 
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/studyApprovals", method = RequestMethod.POST, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Creates the study approval", notes = "This call will create a new study approval. Body contains the ID of the user making the API call for permissions checking and audit logs.", consumes = "application/json", produces = "application/json", response = Integer.class, tags = { "studyApprovals" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = Integer.class) })
	public ResponseEntity<ResponseWrapper> createStudyApproval(@ApiParam(required = true, name = "studyApproval", value = "Study Approval details") @RequestBody StudyApprovalDTO studyApprovalDto) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Create Study Approval");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.createStudyApproval(studyApprovalDto);
		} catch (Exception ex) {
			wrapper.setResponse(studyApprovalDto);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_CREATESA.getErrorMessage(), ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Update study approval.
	 *
	 * @param id
	 *            the id
	 * @param studyApprovalDto
	 *            the study approval
	 * @return the response entity
	 * 
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/studyApprovals/{id}", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Update study approval", notes = "Modify and save changes to a study approval", consumes = "application/json", produces = "application/json", response = Integer.class, tags = { "studyApprovals" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = Integer.class) })
	public ResponseEntity<ResponseWrapper> updateStudyApproval(@ApiParam(required = true, name = "id", value = "Study Approval ID") @PathVariable int id, @ApiParam(required = true, name = "studyApproval", value = "Study Approval details") @RequestBody StudyApprovalDTO studyApprovalDto) throws GenisisException {
		ResponseWrapper wrapper = new ResponseWrapper();
		if (id > 0) {
			studyApprovalDto.setId(id);
			try {
				wrapper = businessService.updateStudyApproval(studyApprovalDto);
			} catch (Exception ex) {
				wrapper.setResponse(studyApprovalDto);
				wrapper.setSuccess(false);
				wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
				LOGGER.error(ErrorEnum.EXP_CTRL_UPDATESA.getErrorMessage(), ex);
			}
			return createReposneWrapper(wrapper);
		} else {
			LOGGER.error("No ID Found");
			wrapper = new ResponseWrapper();
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			return new ResponseEntity<>(wrapper, HttpStatus.NOT_MODIFIED);
		}
	}

	/**
	 * Gets the study approval by id.
	 *
	 * @param id
	 *            the id
	 * @return the study approval by id
	 * 
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/studyApprovals/{id}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Gets the study approval by id", notes = "Returns a list of StudyApproval for a given Study Approval Id", consumes = "application/json", produces = "application/json", response = StudyApprovalDTO.class, tags = { "studyApprovals" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = StudyApprovalDTO.class) })
	public ResponseEntity<ResponseWrapper> getStudyApprovalById(@ApiParam(required = true, name = "id", value = "Study Approval ID") @PathVariable int id) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Study Approval by Id");
		}

		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.getStudyApprovalsByID(id);
		} catch (Exception ex) {
			wrapper.setResponse(id);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_GETSA.getErrorMessage(), id, ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the study approval by user.
	 *
	 * @param uid
	 *            the uid
	 * @return the study approval by user
	 * 
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/studyApprovals/{uid}/{user}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Gets the study approval by user", notes = "Returns a list of StudyApproval for a given User Id", consumes = "application/json", produces = "application/json", response = StudyApprovalDTO.class, tags = { "studyApprovals" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = StudyApprovalDTO.class) })
	public ResponseEntity<ResponseWrapper> getStudyApprovalByUser(@ApiParam(required = true, name = "uid", value = "User ID") @PathVariable String uid) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Study Approval by User");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.getStudyApprovalsByUID(uid);
		} catch (Exception ex) {
			wrapper.setResponse(uid);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_GETSABYUID.getErrorMessage(), uid, ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the all study approvals.
	 *
	 * @return the all study approvals
	 * 
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/studyApprovals", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Gets the all study approvals", notes = "Returns a list of all study approvals in Genisis2", consumes = "application/json", produces = "application/json", response = StudyApprovalDTO[].class, tags = { "studyApprovals" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = StudyApprovalDTO[].class) })
	public ResponseEntity<ResponseWrapper> getAllStudyApprovals() throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("All Study Approvals");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.getStudyApprovals();
		} catch (Exception ex) {
			wrapper.setResponse(null);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_GETALLSAS.getErrorMessage(), ex);
		}
		return createReposneWrapper(wrapper);
	}

	/* C O M M E N T       H I S T O R Y     R E S T      C A L L S */
	/**
	 * Creates the comment history.
	 *
	 * @param commentHistoryDto
	 *            the comment history
	 * @return the response entity
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/comment", method = RequestMethod.POST, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Creates the comment history", notes = "This will create a new study approval", consumes = "application/json", produces = "application/json", response = CommentHistoryDTO.class, tags = { "commentHistory" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = CommentHistoryDTO.class) })
	public ResponseEntity<ResponseWrapper> createCommentHistory(@ApiParam(required = true, name = "commentHistory", value = "CommentHistory details") @RequestBody CommentHistoryDTO commentHistoryDto) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Create Comment History");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.createCommentHistory(commentHistoryDto);
		} catch (Exception ex) {
			wrapper.setResponse(commentHistoryDto);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_CREATECOMMENT.getErrorMessage(), ex);
		}
		return createReposneWrapper(wrapper);
	}

	/* ACTIVITI R E S T C A L L S */
	/**
	 * Gets the request tracking by id.
	 *
	 * @param id  the id
	 * @param  userRole   The user Role
	 * @return the request tracking by id
	 * 
	 * 
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	// Request History
	@RequestMapping(value = "/requests/{id}/{userRole}/log", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Gets the request tracking by id", notes = "Retrives list of request history for a given requestId", consumes = "application/json", produces = "application/json", response = RequestHistoryDTO[].class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = RequestHistoryDTO[].class) })
	public ResponseEntity<ResponseWrapper> getRequestTrackingById(@ApiParam(required = true, name = "id", value = "RequestID") @PathVariable int id, @ApiParam(required = true, name = "userRole", value = "Role") @PathVariable String userRole ) throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Get Request Tracking by Id {} and user Role {} ",id, userRole);
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.getRequestTrackingByID(id, userRole);
		} catch (Exception ex) {
			wrapper.setResponse(null);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_GETREQUESTTRACK.getErrorMessage(), id, ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the request data sources.
	 *
	 * @return the request data sources
	 * 
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/datasources", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Gets the request data sources", notes = "Retrives list of Sources from a GenisisDB", consumes = "application/json", produces = "application/json", response = SourceDTO[].class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = SourceDTO[].class) })
	public ResponseEntity<ResponseWrapper> getRequestDataSources() throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("All Request Data Sources");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.getRequestDataSources();
		} catch (Exception ex) {
			wrapper.setResponse(null);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_GETDS.getErrorMessage(), ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the request types.
	 *
	 * @return the request types
	 * 
	 * @throws GenisisException
	 *             the Genisis exception
	 */
	@RequestMapping(value = "/requestTypes", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ApiOperation(value = "Gets the request types", notes = "Retrives list of request types from a GenisisDB", consumes = "application/json", produces = "application/json", response = RequestType[].class, tags = { "requests" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = RequestType[].class) })
	public ResponseEntity<ResponseWrapper> getRequestTypes() throws GenisisException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("All Request Types");
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			wrapper = businessService.getRequestTypes();
		} catch (Exception ex) {
			wrapper.setResponse(null);
			wrapper.setSuccess(false);
			wrapper.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			LOGGER.error(ErrorEnum.EXP_CTRL_GETREQUESTTYPE.getErrorMessage(), ex);
		}
		return createReposneWrapper(wrapper);
	}

	/**
	 * Creates the reposne wrapper.
	 *
	 * @param wrapper
	 *            the wrapper
	 * @return the response entity
	 */
	private ResponseEntity<ResponseWrapper> createReposneWrapper(ResponseWrapper wrapper) {
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Pragma", "No-cache");
		responseHeaders.add("Cache-Control", "no-cache");
		return new ResponseEntity<>(wrapper, responseHeaders, HttpStatus.OK);
	}

	/**
	 * Exception handler.
	 *
	 * @param ex
	 *            the ex
	 * @return the response entity
	 */
	@ExceptionHandler({ GenisisException.class, Exception.class })
	public ResponseEntity<ErrorResponse> exceptionHandler(Exception ex) {
		LOGGER.error(ErrorEnum.EXCEPTION_OCCURED.getErrorMessage(), ex.getMessage(), ex);
		ErrorResponse error = new ErrorResponse();
		error.setErrorCode(HttpStatus.PRECONDITION_FAILED.value());
		error.setMessage(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
		return new ResponseEntity<>(error, HttpStatus.OK);
	}

	/**
	 * Gets the business service.
	 *
	 * @return the business service
	 */
	public BusinessService getBusinessService() {
		return businessService;
	}

	/**
	 * Sets the business service.
	 *
	 * @param businessService
	 *            the new business service
	 */
	@Autowired
	public void setBusinessService(BusinessService businessService) {
		this.businessService = businessService;
	}
}
